#!/bin/sh

set -e

# Primary.
AI_PORT="11435"
AI_SERVICE_NAME="ollama"
AI_PULL_TIMEOUT="300"
API_PORT="8000"
API_SERVICE_NAME="beecr"
WORK_DIR="/opt/beecr"
LICENSE_NAME="beecr.lic"

# Secondary.
AI_SERVICE_CONFIG_PATH="/etc/systemd/system/${AI_SERVICE_NAME}.service"
AI_SERVICE_EXTRA_CONFIG_DIR="${AI_SERVICE_CONFIG_PATH}.d"
API_SERVICE_CONFIG_PATH="/etc/systemd/system/${API_SERVICE_NAME}.service"
API_SERVICE_EXTRA_CONFIG_DIR="${API_SERVICE_CONFIG_PATH}.d"
VENV="${WORK_DIR}/venv"
PIP3="${VENV}/bin/pip3"

# Log.
INSTALLER_SLUG="BeeCR Installer"
INFO_LOG_PREFIX="INFO\t[${INSTALLER_SLUG}]\t"
WARNING_LOG_PREFIX="WARNING\t[${INSTALLER_SLUG}]\t"
ERROR_LOG_PREFIX="ERROR\t[${INSTALLER_SLUG}]\t"

# Misc.
SCRIPT_DIR="$( cd "$( dirname "${0}" )" && pwd )"



#################################### Options ###################################

# Parse options.
IP="0.0.0.0"
IP_QUALIFIED="${IP}"
if [ "$#" -eq 0 ]; then
    HELP="1"
fi
while getopts ":hiul:6" o; do
    case "${o}" in
        l)  # license
            LICENSE="${OPTARG}"
            if [ -z "${LICENSE}" ]; then
                echo "${ERROR_LOG_PREFIX}❌ The path to the license file is empty."
                exit 1
            fi
            ;;
        i)  # install
            INSTALL="1"
            ;;
        u)  # uninstall
            UNINSTALL="1"
            ;;
        6)  # Use IPv6 instead of IPv4.
            IP="::"
            IP_QUALIFIED="[${IP}]"
            ;;
        h)  # help
            HELP="1"
            ;;
        *)  # do nothing with other flags
            ;;
    esac
done
shift $((OPTIND-1))

# Help.
if [ -n "${HELP}" ]; then
    echo "SYNOPSIS:"
    echo "   ${0} [-h] [-i] [-u] [-l LICENSE]"
    echo ""
    echo "DESCRIPTION:"
    echo "   Install and configure BeeCR with all dependencies."
    echo ""
    echo "   The following options are available:"
    echo ""
    echo "   -h    Show this help message and exit."
    echo ""
    echo "   -i    Installation action."
    echo "         Can not be used with \"-u\" simultaneously."
    echo ""
    echo "   -6    Use IPv6 instead of IPv4."
    echo ""
    echo "   -u    Uninstallation action."
    echo "         Can not be used with \"-i\" simultaneously."
    echo ""
    echo "   -l LICENSE"
    echo "         If passed, the installer will also place your LICENSE file"
    echo "         in the correct place. Meaningless for uninstallation action."
    echo ""
    echo "         If you don not have a license at this moment, simply skip"
    echo "         passing this option. Later, place your license file named"
    echo "         \"${LICENSE_NAME}\" in the \"${WORK_DIR}\" directory"
    echo "         and restart the \"${API_SERVICE_NAME}\" system service."
    echo ""
    echo "REQUIRES:"
    echo "   * \"Linux\" with non \"WSL\" kernel"
    echo "   * \"x86_64\" architecture"
    echo "   * \"curl\" tool"
    echo "   * \"python3\" interpreter (3.11+)"
    echo "   * \"virtualenv\" Python virtual environment creator"
    echo "   * NVidia GPU"
    echo ""
    exit 0
fi

# Check that action is passed.
if [ -z "${INSTALL}" ] && [ -z "${UNINSTALL}" ]; then
    echo "${ERROR_LOG_PREFIX}❌ No action passed."
    exit 1
elif [ -n "${INSTALL}" ] && [ -n "${UNINSTALL}" ]; then
    echo "${ERROR_LOG_PREFIX}❌ You cannot mix installation and uninstallation actions at the same time."
    exit 1
fi



##################################### Sudo #####################################

SUDO=
if [ "$(id -u)" -ne 0 ]; then
    echo "${WARNING_LOG_PREFIX}🟡 Current user is not root. This script will ask for superuser permissions."
    SUDO="sudo"
else
    echo "${INFO_LOG_PREFIX}✅ Superuser."
fi



################################### Uninstall ##################################

if [ -n "${UNINSTALL}" ]; then
    echo "${INFO_LOG_PREFIX}👉 Uninstall..."

    { ${SUDO} systemctl stop "${API_SERVICE_NAME}" || true; } && sleep 5s
    { ${SUDO} systemctl disable "${API_SERVICE_NAME}" || true; } && sleep 1
    { ${SUDO} systemctl stop "${AI_SERVICE_NAME}" || true; } && sleep 5
    { ${SUDO} systemctl disable "${AI_SERVICE_NAME}" || true; } && sleep 1
    ${SUDO} rm -rf "${AI_SERVICE_CONFIG_PATH}" "${AI_SERVICE_EXTRA_CONFIG_DIR}"
    ${SUDO} rm -rf "${API_SERVICE_CONFIG_PATH}" "${API_SERVICE_EXTRA_CONFIG_DIR}"
    ${SUDO} systemctl daemon-reload && sleep 1
    if command -v "${AI_SERVICE_NAME}" >/dev/null 2>&1; then
        ${SUDO} rm -rf "$(which ${AI_SERVICE_NAME})"
    fi
    if [ -e "/usr/share/${AI_SERVICE_NAME}" ]; then
        ${SUDO} rm -r "/usr/share/${AI_SERVICE_NAME}"
    fi

    { ${SUDO} userdel ${AI_SERVICE_NAME} || true; } && sleep 1
    { ${SUDO} groupdel ${AI_SERVICE_NAME} || true; } && sleep 1

    if [ -e "${WORK_DIR}/logging.json" ]; then
        ${SUDO} rm -r "${WORK_DIR}/logging.json"
    fi
    if [ -e "${VENV}" ]; then
        ${SUDO} rm -r "${VENV}"
    fi

    echo "${INFO_LOG_PREFIX}✅ Uninstalled."

    if [ -f "${WORK_DIR}/${LICENSE_NAME}" ]; then
        echo "${WARNING_LOG_PREFIX}🟡 License file \"${WORK_DIR}/${LICENSE_NAME}\" exists. If necessary, consider removing it manually."
    fi

    exit 0
fi



################################ Install: Checks ###############################

echo "${INFO_LOG_PREFIX}👉 Step 1 – Check environment..."

# Check Linux.
OS=$(uname -s)
if [ "${OS}" != "Linux" ]; then
    echo "${ERROR_LOG_PREFIX}❌ Unsupported OS: \"${OS}\". This script is intended to run on \"Linux\" only."
    exit 1
fi
echo "${INFO_LOG_PREFIX}✅ OS is Linux."

# Check arch.
ARCH=$(uname -m)
if [ "${ARCH}" != "x86_64" ]; then
    echo "${ERROR_LOG_PREFIX}❌ Unsupported architecture: \"${ARCH}\". This script is intended to run on \"x86_64\" architecture only."
    exit 1
fi
echo "${INFO_LOG_PREFIX}✅ Architecture is \"x86_64\"."

# Check kernel.
WSL=$(uname -r | { grep '.*icrosoft.*' || true; } )
if [ -n "${WSL}" ]; then
    echo "${ERROR_LOG_PREFIX}❌ \"Microsoft WSL\" is not supported."
    exit 1
fi
echo "${INFO_LOG_PREFIX}✅ Kernel is non-WSL."

# Check curl.
if ! command -v "curl" >/dev/null 2>&1; then
    echo "${ERROR_LOG_PREFIX}❌ \"cURL\" tool is not available. Please, install it."
    exit 1
fi
echo "${INFO_LOG_PREFIX}✅ \"cURL\" is available."

# Check Python and virtualenv.
if ! command -v "virtualenv" >/dev/null 2>&1; then
    echo "${ERROR_LOG_PREFIX}❌ The \"virtualenv\" (Python virtual environment creator) is not available. You can install it, for example, using your system package manager (search for the \"python3-virtualenv\" package or similar)."
    exit 1
else
    PYTHON3VGOOD=$(python3 -c 'import sys; print(sys.version_info.major==3 and sys.version_info.minor>=11)')
    if [ "${PYTHON3VGOOD}" != "True" ]; then
        echo "${ERROR_LOG_PREFIX}❌ Python 3.11 (or higher) is required."
        echo "${INFO_LOG_PREFIX}Note:
    Installing Python 3.11 alongside older versions of Python may be tricky and
    highly dependent on your Linux distribution.

    Once you have installed Python 3.11 (e.g., using your system package manager),
    consider updating symbolic links to your Python interpreter and shared libraries.

    For example, on Ubuntu 20.04 with Python 3.8 preinstalled, you may need to
    run the following commands as a superuser:
        ${SUDO} update-alternatives --install /usr/bin/python3 python3 /usr/bin/python3.8 1
        ${SUDO} update-alternatives --install /usr/bin/python3 python3 /usr/bin/python3.11 2
        ${SUDO} update-alternatives --install /usr/bin/python python /usr/bin/python3 1
        ${SUDO} ln -s /usr/lib/python3/dist-packages/apt_pkg{.cpython-38-x86_64-linux-gnu,}.so"
        exit 1
    fi
fi
echo "${INFO_LOG_PREFIX}✅ \"Python 3\" and \"virtualenv\" are available."



############################## Install: AI Server ##############################

# Install AI server.
echo "${INFO_LOG_PREFIX}👉 Step 2 – Install AI server..."
curl -fsSL https://ollama.com/install.sh | OLLAMA_VERSION="0.3.14" sh
hash -r
if ! command -v "${AI_SERVICE_NAME}" >/dev/null 2>&1; then
    echo "${ERROR_LOG_PREFIX}❌ Failed to install the \"AI server\"."
    exit 1
fi
echo "${INFO_LOG_PREFIX}✅ AI server installed."

# Configure AI server.
echo "${INFO_LOG_PREFIX}👉 Step 3 – Configure AI server..."
${SUDO} mkdir -p "${AI_SERVICE_EXTRA_CONFIG_DIR}"
${SUDO} sh -c "cat > '${AI_SERVICE_EXTRA_CONFIG_DIR}/env.conf' << EOF
[Service]
Environment=\"OLLAMA_HOST=${IP_QUALIFIED}:${AI_PORT}\"
Environment=\"OLLAMA_KEEP_ALIVE=-1\"
Environment=\"OLLAMA_NUM_PARALLEL=2\"
EOF"
${SUDO} systemctl daemon-reload && sleep 1
${SUDO} systemctl enable "${AI_SERVICE_NAME}" && sleep 1
${SUDO} systemctl restart "${AI_SERVICE_NAME}" && sleep 5
AI_HTTP_RESP=$({ curl -s -o /dev/null -w "%{http_code}" "http://localhost:${AI_PORT}" || true; })
if [ "${AI_HTTP_RESP}" -ne 200 ]; then
    echo "${ERROR_LOG_PREFIX}❌ Failed to configure the \"AI server\"."
    exit 1
fi
echo "${INFO_LOG_PREFIX}✅ AI server configured."

# Pull AI model.
echo "${INFO_LOG_PREFIX}👉 Step 4 – Pull AI model..."
set +e
OLLAMA_HOST="${IP_QUALIFIED}:${AI_PORT}" timeout "${AI_PULL_TIMEOUT}" ollama pull "codestral:22b"  # try with timeout
OLLAMA_PULL_EXIT_CODE="$?"
if [ "${OLLAMA_PULL_EXIT_CODE}" -ne 0 ]; then
    echo "${WARNING_LOG_PREFIX}🟡 It seems like pulling is taking too long. Will retry..."
    ${SUDO} systemctl restart "${AI_SERVICE_NAME}" && sleep 5
    OLLAMA_HOST="${IP_QUALIFIED}:${AI_PORT}" timeout "${AI_PULL_TIMEOUT}" ollama pull "codestral:22b"  # retry again with timeout
    OLLAMA_PULL_EXIT_CODE="$?"
    if [ "${OLLAMA_PULL_EXIT_CODE}" -ne 0 ]; then
        echo "${WARNING_LOG_PREFIX}🟡 It seems like pulling is taking too long. Will retry..."
        OLLAMA_HOST="${IP_QUALIFIED}:${AI_PORT}" ollama pull "codestral:22b"  # retry again without timeout
    fi
fi
set -e

AI_HTTP_RESP_MODEL=$({ curl -s -o /dev/null -w "%{http_code}" "http://${IP_QUALIFIED}:${AI_PORT}/api/show" -d '{"name": "codestral:22b"}' || true; })
if [ "${AI_HTTP_RESP_MODEL}" -ne 200 ]; then
    echo "${ERROR_LOG_PREFIX}❌ Failed to configure the \"AI server\"."
    exit 1
fi
timeout 15 curl -s "http://${IP_QUALIFIED}:${AI_PORT}/api/generate" -d '{"model": "codestral:22b", "prompt": "hello", "stream": false}' >/dev/null 2>&1  # warmup
echo "${INFO_LOG_PREFIX}✅ AI model pulled."



############################## Install: API Server #############################

# Virtual environment
${SUDO} virtualenv "${VENV}"

# Uvicorn.
echo "${INFO_LOG_PREFIX}👉 Step 5 – Install ASGI server to serve API..."
${SUDO} ${PIP3} install 'uvicorn~=0.30'
UVICORN=$(${SUDO} ${PIP3} freeze | { grep 'uvicorn' || true; } )
if [ -z "${UVICORN}" ]; then
    echo "${ERROR_LOG_PREFIX}❌ Failed to install \"ASGI server\"."
    exit 1
fi
echo "${INFO_LOG_PREFIX}✅ \"ASGI server\" installed."

# API server package.
echo "${INFO_LOG_PREFIX}👉 Step 6 – Install API server..."
PYTHON_TAG=$(python3 -c "import sysconfig; print(sysconfig.get_python_version().replace('.', ''), end='')")
${SUDO} ${PIP3} install --force-reinstall "${SCRIPT_DIR}/beecr_"*"${PYTHON_TAG}"*"linux_x86_64.whl"
API_PACKAGE=$(${SUDO} ${PIP3} freeze | { grep 'beecr.*' || true; } )
if [ -z "${API_PACKAGE}" ]; then
    echo "${ERROR_LOG_PREFIX}❌ Failed to install \"API server\"."
    exit 1
fi
echo "${INFO_LOG_PREFIX}✅ \"API server\" installed."

# API server configuration.
echo "${INFO_LOG_PREFIX}👉 Step 7 – Configure API server..."
${SUDO} mkdir -p "${WORK_DIR}"
${SUDO} mkdir -p "${API_SERVICE_EXTRA_CONFIG_DIR}"
${SUDO} sh -c "cat > '${API_SERVICE_CONFIG_PATH}' << EOF
[Unit]
Description=BeeCR Service
After=network-online.target

[Service]
WorkingDirectory=${WORK_DIR}
ExecStart=${VENV}/bin/python3 -m uvicorn --host ${IP} --port \\\$PORT --log-config ${WORK_DIR}/logging.json beecr.api.main:app
Restart=always
RestartSec=3
Environment=\"PATH=/usr/local/sbin:/usr/local/bin:/usr/sbin:/usr/bin:/sbin:/bin:/snap/bin\"
Environment=\"PORT=${API_PORT}\"

[Install]
WantedBy=default.target
EOF"
${SUDO} sh -c "cat > '${API_SERVICE_EXTRA_CONFIG_DIR}/env.conf' << EOF
[Service]

# Default GitLab host and token.
#
# Will be used only if the requests do not pass these parameters explicitly.
# Environment=\"GITLAB_HOST=https://gitlab.com\"
# Environment=\"GITLAB_TOKEN=\"


# Ollama-compatible API settings.
Environment=\"OLLAMA_HOST=http://localhost:${AI_PORT}\"


# Expected GPU memory in GB for Ollama.
#
# This is essential to define the proper amount of GPU memory on the
# AI server when using Ollama-compatible models.
#
# If your AI server has multiple GPUs, you should pass the sum of
# all GPUs memory.
Environment=\"OLLAMA_GPU_MEM=24\"


# Expected number of parallel threads for Ollama.
#
# This is essential to define the proper number of parallel
# processing threads when using Ollama-compatible models.
#
# Keep this option synchronized between the AI and API servers.
Environment=\"OLLAMA_NUM_PARALLEL=2\"


# OpenAI-compatible API settings.
#
# Edit if you wish to use AI models via OpenAI-compatible API.
# Environment=\"OPENAI_HOST=https://api.openai.com\"
# Environment=\"OPENAI_API_KEY=\"


# Default AI model name.
#
# This AI model will be used for handling API requests without
# an explicit model name passed.
#
# For OpenAI-compatible APIs use model names as they are,
# e.g. \"gpt-4o\".
#
# For Ollama-compatible APIs use model names with the prefix \"ollama/\",
# e.g. \"ollama/codestral:22b\".
Environment=\"MODEL=ollama/codestral:22b\"
EOF"
${SUDO} sh -c "cat > '${WORK_DIR}/logging.json' << EOF
{
    \"version\": 1,
    \"filters\": {
        \"correlation_id\": {
            \"()\": \"asgi_correlation_id.CorrelationIdFilter\",
            \"uuid_length\": 32,
            \"default_value\": \"-\"
        }
    },
    \"formatters\": {
        \"default\": {
            \"format\": \"%(levelname)s\t[beecr-%(correlation_id)s]\t%(message)s\",
            \"datefmt\": \"%Y-%m-%d %H:%M:%S\"
        }
    },
    \"handlers\": {
        \"console\": {
            \"formatter\": \"default\",
            \"filters\": [\"correlation_id\"],
            \"class\": \"logging.StreamHandler\",
            \"stream\": \"ext://sys.stdout\",
            \"level\": \"DEBUG\"
            }
    },
    \"root\": {
            \"handlers\": [\"console\"],
            \"level\": \"INFO\"
    },
    \"loggers\": {
        \"uvicorn\": {
            \"propagate\": true
        },
        \"uvicorn.access\": {
            \"propagate\": true
        }
    }
}
EOF"

${SUDO} systemctl daemon-reload && sleep 1
${SUDO} systemctl enable "${API_SERVICE_NAME}" && sleep 1
{ ${SUDO} systemctl stop "${API_SERVICE_NAME}" || true; } && sleep 5
if [ -n "${LICENSE}" ]; then
    ${SUDO} cp "${LICENSE}" "${WORK_DIR}/${LICENSE_NAME}"
fi
if [ -f "${WORK_DIR}/${LICENSE_NAME}" ]; then
    ${SUDO} systemctl start "${API_SERVICE_NAME}" && sleep 5
    API_HTTP_RESP_VERSION=$({ curl -s -o /dev/null -w "%{http_code}" "http://localhost:${API_PORT}/version" || true; })
    if [ "${API_HTTP_RESP_VERSION}" -ne 200 ]; then
        echo "${ERROR_LOG_PREFIX}❌ Failed to configure the \"API server\"."
        echo "${INFO_LOG_PREFIX}Note: Use \"journalctl\" to view logs of the ${API_SERVICE_NAME} service. Example: \"${SUDO} journalctl -fu ${API_SERVICE_NAME}\"."
        exit 1
    fi
else
    echo "${WARNING_LOG_PREFIX}🟡 API server license is missing. Put the license file \"${LICENSE_NAME}\" to the \"${WORK_DIR}\" directory and restart the service using command \"systemctl restart ${API_SERVICE_NAME}\"."
    NVSMIL="$(nvidia-smi -L)"
    if [ -n "${NVSMIL}" ]; then
        echo "${INFO_LOG_PREFIX}Note: To request a license file, please provide this information to info@cvisionlab.com: \"${NVSMIL}\"."
    fi
    exit 0
fi

echo "${INFO_LOG_PREFIX}✅ \"API server\" configured."
echo "${INFO_LOG_PREFIX}Note: Use \"journalctl\" to view logs of the ${API_SERVICE_NAME} service. Example: \"${SUDO} journalctl -fu ${API_SERVICE_NAME}\"."
